#include "led.h"
#include <stddef.h>
/* LED引脚映射表 */
static const uint32_t LED_PINS[LED_COUNT] = {
    LED_Key_PIN,    // LED_KEY
    LED_Err_PIN     // LED_ERR
};
/* KEY灯闪烁控制结构体 */
typedef struct {
    uint32_t lastToggleTime;  // 上次状态切换时间戳
    uint32_t interval;        // 闪烁间隔(ms)
    uint8_t isActive;         // 闪烁使能标志
} LED_KEY_Blink_Control;

/* ERR灯闪烁控制结构体 */ 
typedef struct {
    uint32_t lastToggleTime;  // 上次状态切换时间戳
    uint32_t interval;        // 闪烁间隔(ms)
    uint8_t isActive;         // 闪烁使能标志
} ERR_Blink_Ctrl_t;

/* 静态全局变量 */
static LED_KEY_Blink_Control keyBlinkCtrl = {0};
static ERR_Blink_Ctrl_t errBlinkCtrl = {0};


/**
 * @brief 关闭指定LED
 * @param num LED编号(LED_Number枚举值)
 */
void LED_off(LED_Number num)
{
    if (num >= LED_COUNT) return;
    DL_GPIO_clearPins(GPIOA, LED_PINS[num]);
}

/**
 * @brief 点亮指定LED  
 * @param num LED编号(LED_Number枚举值)
 */
void LED_on(LED_Number num)
{
    if (num >= LED_COUNT) return;
    DL_GPIO_setPins(GPIOA, LED_PINS[num]);
}

/**
 * @brief 切换指定LED状态
 * @param num LED编号(LED_Number枚举值) 
 */
void LED_toggle(LED_Number num)
{
    if (num >= LED_COUNT) return;
    DL_GPIO_togglePins(GPIOA, LED_PINS[num]);
}

/******************** KEY灯闪烁控制 ********************/

/**
 * @brief 初始化KEY灯闪烁功能
 * @param interval 闪烁间隔时间(ms)
 */
void LED_KEY_Blink_Init(uint32_t interval)
{
    keyBlinkCtrl.interval = interval;
    keyBlinkCtrl.lastToggleTime = Tick;
    keyBlinkCtrl.isActive = 1;
    LED_off(LED_KEY);
}

/**
 * @brief 更新KEY灯闪烁状态(需在主循环调用)
 */
void LED_KEY_Blink_Update(void)
{
    if (!keyBlinkCtrl.isActive) return;
    
    if ((Tick - keyBlinkCtrl.lastToggleTime) >= keyBlinkCtrl.interval) {
        LED_toggle(LED_KEY);
        keyBlinkCtrl.lastToggleTime = Tick;
    }
}

/**
 * @brief 设置KEY灯闪烁间隔
 * @param newInterval 新的闪烁间隔(ms)
 */
void LED_KEY_Blink_SetInterval(uint32_t newInterval)
{
    keyBlinkCtrl.interval = newInterval;
}

/**
 * @brief 停止KEY灯闪烁
 */
void LED_KEY_Blink_Stop(void)
{
    keyBlinkCtrl.isActive = 0;
    LED_off(LED_KEY);
}

/**
 * @brief 开始KEY灯闪烁
 */
void LED_KEY_Blink_Start(void)
{
    keyBlinkCtrl.isActive = 1;
    keyBlinkCtrl.lastToggleTime = Tick;
}

/******************** ERR灯闪烁控制 ********************/

/**
 * @brief 初始化ERR灯闪烁功能
 * @param interval 闪烁间隔时间(ms)
 */
void LED_ERR_Blink_Init(uint32_t interval)
{
    errBlinkCtrl.interval = interval;
    errBlinkCtrl.lastToggleTime = Tick;
    errBlinkCtrl.isActive = 1;
    LED_off(LED_ERR);
}

/**
 * @brief 更新ERR灯闪烁状态(需在主循环调用)
 */
void LED_ERR_Blink_Update(void)
{
    if (!errBlinkCtrl.isActive) return;
    
    if ((Tick - errBlinkCtrl.lastToggleTime) >= errBlinkCtrl.interval) {
        LED_toggle(LED_ERR);
        errBlinkCtrl.lastToggleTime = Tick;
    }
}

/**
 * @brief 设置ERR灯闪烁间隔
 * @param newInterval 新的闪烁间隔(ms)
 */
void LED_ERR_Blink_SetInterval(uint32_t newInterval)
{
    errBlinkCtrl.interval = newInterval;
}

/**
 * @brief 停止ERR灯闪烁
 */
void LED_ERR_Blink_Stop(void)
{
    errBlinkCtrl.isActive = 0;
    LED_off(LED_ERR);
}

/**
 * @brief 开始ERR灯闪烁
 */
void LED_ERR_Blink_Start(void)
{
    errBlinkCtrl.isActive = 1;
    errBlinkCtrl.lastToggleTime = Tick;
}

/**
 * @brief 检查ERR灯闪烁是否激活
 * @return 1-激活 0-未激活
 */
uint8_t LED_ERR_Blink_IsActive(void)
{
    return errBlinkCtrl.isActive;
}